from pyparsing import Literal

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector
from codeable_detectors.detector_context import DetectorContext
from codeable_detectors.evidences import FailedEvidence, ComponentEvidence
from codeable_detectors.pyparsing_patterns import ID


def detect_docker_compose_image(ctx, image_name_prefix):
    matches = []
    services_block_matches = ctx.match_indented_block(Literal("services") + Literal(":"))
    for services_block_match in services_block_matches:
        service_spec_matches = DetectorContext(services_block_match).match_indented_block(
            ID + Literal(":"))
        for service_spec_match in service_spec_matches:
            image_results = DetectorContext(service_spec_match).matches_pattern(Literal("image") +
                                                                                Literal(":") + Literal(
                image_name_prefix))
            if image_results:
                matches.append(services_block_match)
                break
    return matches


class DockerComposeImageBasedComponent(AtLeastOneFileMatchesDetector):
    def __init__(self, image_name_prefix):
        super().__init__()
        self.image_name_prefix = image_name_prefix
        self.file_endings = ["yaml"]

    def detect_in_context(self, ctx, **kwargs):
        matches = detect_docker_compose_image(ctx, self.image_name_prefix)
        if not matches:
            return FailedEvidence("docker compose image '" + self.image_name_prefix +
                                  "' not used in a services block")
        return ComponentEvidence(matches).set_properties(kwargs=kwargs)


class DockerComposeRabbitMQImageBasedComponent(DockerComposeImageBasedComponent):
    def __init__(self):
        super().__init__("rabbitmq")

    def detect_in_context(self, ctx, **kwargs):
        evidence = super().detect_in_context(ctx, **kwargs)
        if not evidence.has_succeeded():
            return evidence
        return evidence.set_properties(detector_name="Rabbit MQ",
                                       detector_component_types="messageBroker", detector_technology_types="rabbitmq")


class DockerComposeMySQLDBImageBasedComponent(DockerComposeImageBasedComponent):
    def __init__(self):
        super().__init__("mysql")

    def detect_in_context(self, ctx, **kwargs):
        evidence = super().detect_in_context(ctx, **kwargs)
        if not evidence.has_succeeded():
            return evidence
        return evidence.set_properties(detector_name="MySQL DB",
                                       detector_component_types="mySQLDB", detector_technology_types="mysql")
